-- FuelConsumptionHUD (FS25, descVersion 104)
-- Minimal build: stable bottom-right corner anchoring with UI-scale-aware offsets.

FuelConsumptionHUD = {
    REFRESH_MS = 250.0,
    DIESEL_DISPLAY_PERIOD_SEC = 1.0,

    -- Electric (kept minimal for EV support)
    ELECTRIC_IDLE_SPEED_KMH = 0.2,
    ELECTRIC_DEADBAND_KW = 0.15,

    -- Diesel
    DIESEL_DEADBAND_LPH = 0.05,
    DIESEL_IDLE_FLOOR_LPH = 0.6,
    DIESEL_MAX_VALID_LPH = 1200.0,

    -- Fill-level delta estimator anti-spike
    FILL_DELTA_MIN_ELAPSED_SEC = 1.0,
    FILL_DELTA_MIN_CHANGE_L = 0.05,
    FILL_DELTA_SMOOTH_ALPHA = 0.35,

    -- UI: adjust these to place/size the HUD
    UI_SIZE = 0.014,                        -- base text height
    UI_CORNER_OFFSET_PX = { x = 10, y = 186 }, -- inward offset from bottom-right corner (pixels), scales with HUD scale

    -- If the engine already scales text returned by getCorrectTextSize/renderText, set true to avoid double scaling
    UI_ENGINE_APPLIES_UI_SCALE = false,

    -- Show even when base HUD is hidden
    SHOW_WITH_HUD_HIDDEN = true,
}

local S = {
    accRefresh = 0.0,
    visible = false,
    text = "",

    lastVehicle = nil,

    dieselText = nil,
    lastDieselShownT = 0.0,

    energyType = "OTHER",
    fuelUnitIndex = nil,
    wasMotorOn = false,

    -- Event-based fill-level estimator
    lvlSampleLevel = nil,
    lvlSampleTime = nil,
    dieselRateLph = 0.0,
}

local function nowSec() return (g_time or 0) * 0.001 end
local function clamp(x, a, b) return math.max(a, math.min(b, x)) end
local function clamp01(x) return clamp(x, 0, 1) end

-- Get in-game UI scale from settings with fallbacks
local function getUiScale()
    local scale = 1.0
    if g_gameSettings ~= nil then
        if g_gameSettings.getValue ~= nil then
            local ok, val = pcall(function() return g_gameSettings:getValue("uiScale") end)
            if ok and type(val) == "number" and val > 0 then
                scale = val
            end
        end
        if scale == 1.0 and type(g_gameSettings.uiScale) == "number" and g_gameSettings.uiScale > 0 then
            scale = g_gameSettings.uiScale
        end
    end
    if scale == 1.0 and type(_G["g_uiScale"]) == "number" and _G["g_uiScale"] > 0 then
        scale = _G["g_uiScale"]
    end
    return scale
end

-- Vehicle detection (compact and robust)
local function getCurrentVehicle()
    local hud = g_currentMission and g_currentMission.hud
    local speedo = hud and hud.speedMeter
    if speedo and speedo.vehicle ~= nil then return speedo.vehicle end
    if hud and hud.activeDisplayVehicle ~= nil then return hud.activeDisplayVehicle end
    if g_currentMission and g_currentMission.controlledVehicle ~= nil then return g_currentMission.controlledVehicle end
    if g_currentMission and g_currentMission.vehicles then
        for _, v in ipairs(g_currentMission.vehicles) do
            if v then
                if v.getIsControlled and v:getIsControlled() then return v end
                if v.getIsEntered and v:getIsEntered() then return v end
                local spec = v.spec_enterable
                if spec and (spec.isControlled or spec.entered or (spec.numPassengers or 0) > 0) then
                    return v
                end
            end
        end
    end
    return nil
end

-- Resolve energy level/type and fill unit index
local function resolveFillUnitIndex(root, ft)
    if root == nil or ft == nil then return nil end
    if root.getConsumerFillUnitIndex ~= nil then
        local ok, idx = pcall(function() return root:getConsumerFillUnitIndex(ft) end)
        if ok and idx ~= nil then return idx end
    end
    return nil
end

local function getEnergyLevelAndTypeWithIndex(veh)
    if veh == nil then return nil, "OTHER", nil, nil, nil end
    local root = veh.rootVehicle or veh
    local specFill = root.spec_fillUnit
    local specMotor = root.spec_motorized
    if specFill == nil or specFill.fillUnits == nil then
        return nil, "OTHER", nil, specMotor, specFill
    end

    local FT = _G["FillType"] or {}
    local idx

    if FT.ELECTRICCHARGE then
        idx = resolveFillUnitIndex(root, FT.ELECTRICCHARGE)
        if idx and specFill.fillUnits[idx] then
            return specFill.fillUnits[idx].fillLevel, "ELECTRIC", idx, specMotor, specFill
        end
    end
    if FT.ELECTRICITY then
        idx = resolveFillUnitIndex(root, FT.ELECTRICITY)
        if idx and specFill.fillUnits[idx] then
            return specFill.fillUnits[idx].fillLevel, "ELECTRIC", idx, specMotor, specFill
        end
    end
    if FT.DIESEL then
        idx = resolveFillUnitIndex(root, FT.DIESEL)
        if idx and specFill.fillUnits[idx] then
            return specFill.fillUnits[idx].fillLevel, "LIQUID", idx, specMotor, specFill
        end
    end
    if FT.METHANE then
        idx = resolveFillUnitIndex(root, FT.METHANE)
        if idx and specFill.fillUnits[idx] then
            return specFill.fillUnits[idx].fillLevel, "LIQUID", idx, specMotor, specFill
        end
    end

    local motorIdx = specMotor and specMotor.fuelFillUnitIndex or nil
    if motorIdx and specFill.fillUnits[motorIdx] then
        local fu = specFill.fillUnits[motorIdx]
        if (FT.ELECTRICCHARGE and fu.fillType == FT.ELECTRICCHARGE) or (FT.ELECTRICITY and fu.fillType == FT.ELECTRICITY) then
            return fu.fillLevel, "ELECTRIC", motorIdx, specMotor, specFill
        end
        if (FT.DIESEL and fu.fillType == FT.DIESEL) or (FT.METHANE and fu.fillType == FT.METHANE) then
            return fu.fillLevel, "LIQUID", motorIdx, specMotor, specFill
        end
    end

    for i, fu in ipairs(specFill.fillUnits) do
        if fu then
            if ((FT.ELECTRICCHARGE and fu.fillType == FT.ELECTRICCHARGE) or (FT.ELECTRICITY and fu.fillType == FT.ELECTRICITY)) then
                return fu.fillLevel, "ELECTRIC", i, specMotor, specFill
            end
        end
    end
    for i, fu in ipairs(specFill.fillUnits) do
        if fu then
            if ((FT.DIESEL and fu.fillType == FT.DIESEL) or (FT.METHANE and fu.fillType == FT.METHANE)) then
                return fu.fillLevel, "LIQUID", i, specMotor, specFill
            end
        end
    end

    return nil, "OTHER", nil, specMotor, specFill
end

-- Helpers
local function getSpeedKmh(veh)
    local s = 0.0
    if veh and veh.getLastSpeed ~= nil then
        local ok, val = pcall(function() return veh:getLastSpeed() end)
        if ok and val then s = val end
    end
    return s or 0.0
end

local function isMotorLikelyRunning(specMotor, veh)
    if not specMotor then return false end
    if specMotor.isMotorStarted == true then return true end
    if specMotor.getIsMotorStarted ~= nil then
        local ok, started = pcall(function() return specMotor:getIsMotorStarted() end)
        if ok and started then return true end
    end
    local m = specMotor.motor
    if m then
        local lastR = (m.lastMotorRpm or 0)
        local minR = (m.minRpm or 0)
        if lastR > 0 and lastR >= minR then return true end
    end
    if getSpeedKmh(veh) > 0.2 then return true end
    return false
end

-- Event-driven fill-level delta estimator to avoid quantization spikes
local function updateFillDeltaEstimator(level)
    local t = nowSec()
    if level == nil then
        S.lvlSampleLevel = nil
        S.lvlSampleTime = nil
        S.dieselRateLph = 0.0
        return 0.0
    end
    if S.lvlSampleLevel == nil or S.lvlSampleTime == nil then
        S.lvlSampleLevel = level
        S.lvlSampleTime = t
        S.dieselRateLph = 0.0
        return 0.0
    end
    local dLevel = S.lvlSampleLevel - level
    local elapsed = t - S.lvlSampleTime

    if math.abs(dLevel) >= FuelConsumptionHUD.FILL_DELTA_MIN_CHANGE_L and elapsed >= FuelConsumptionHUD.FILL_DELTA_MIN_ELAPSED_SEC then
        local used = math.max(0.0, dLevel)
        local instLph = (used / elapsed) * 3600.0
        instLph = math.max(0.0, instLph)
        if instLph > FuelConsumptionHUD.DIESEL_MAX_VALID_LPH then
            instLph = FuelConsumptionHUD.DIESEL_MAX_VALID_LPH
        end
        local alpha = clamp01(FuelConsumptionHUD.FILL_DELTA_SMOOTH_ALPHA)
        S.dieselRateLph = S.dieselRateLph * (1 - alpha) + instLph * alpha
        S.lvlSampleLevel = level
        S.lvlSampleTime = t
    elseif elapsed >= 5.0 then
        S.dieselRateLph = S.dieselRateLph * 0.8
        S.lvlSampleLevel = level
        S.lvlSampleTime = t
    end

    return S.dieselRateLph or 0.0
end

-- Consumer-based estimators
local function arrayContains(t, val)
    for _, v in ipairs(t) do if v == val then return true end end
    return false
end

local function getConsumer(specMotor, candidates)
    if not specMotor then return nil end
    local byName = specMotor.consumersByFillTypeName
    if byName ~= nil then
        for _, name in ipairs(candidates) do
            local c = byName[name]
            if c and (c.usage ~= nil) and c.usage ~= 0 then return c end
        end
    end
    local any = specMotor.consumers
    if type(any) == "table" then
        for _, c in pairs(any) do
            local n = c and (c.fillTypeName or c.name)
            if n and arrayContains(candidates, n) and (c.usage ~= nil) and c.usage ~= 0 then
                return c
            end
        end
        for _, c in pairs(any) do
            if c and (c.usage ~= nil) and c.usage ~= 0 then
                return c
            end
        end
    end
    return nil
end

local function computeElectricUsageFromConsumers(rootVehicle, specMotor, dtMs)
    if specMotor == nil or dtMs == nil or dtMs <= 0 then return nil end
    local m = specMotor.motor
    if not m or m.minRpm == nil or m.maxRpm == nil then return nil end
    local rpmPct = (m.maxRpm ~= m.minRpm) and ((m.lastMotorRpm - m.minRpm) / (m.maxRpm - m.minRpm)) or 0
    rpmPct = clamp01(rpmPct)
    local idleFactor = 0.5
    local rpmFactor = idleFactor + rpmPct * (1 - idleFactor)
    local loadFactor = math.max((specMotor.smoothedLoadPercentage or 0) * rpmPct, 0)
    local motorFactor = 0.5 * (0.2 * rpmFactor + 1.8 * loadFactor)
    local usageFactor = 1.5
    local mi = g_currentMission and g_currentMission.missionInfo
    if mi and mi.fuelUsage == 1 then usageFactor = 1.0 elseif mi and mi.fuelUsage == 3 then usageFactor = 2.5 end
    local consumer = getConsumer(specMotor, {"ELECTRICCHARGE","ELECTRICITY"})
    if consumer == nil then return nil end
    local usage = math.abs(consumer.usage or 0)
    if usage == 0 then return nil end
    local used = usageFactor * motorFactor * usage * dtMs
    local kWhPerHour = (used / dtMs) * 1000.0 * 3600.0
    return kWhPerHour
end

local function computeLiquidUsageFromConsumers(specMotor, dtMs)
    if specMotor == nil or dtMs == nil or dtMs <= 0 then return nil end
    local m = specMotor.motor
    if not m or m.minRpm == nil or m.maxRpm == nil then return nil end
    local rpmPct = (m.maxRpm ~= m.minRpm) and ((m.lastMotorRpm - m.minRpm) / (m.maxRpm - m.minRpm)) or 0
    rpmPct = clamp01(rpmPct)
    local idleFactor = 0.5
    local rpmFactor = idleFactor + rpmPct * (1 - idleFactor)
    local loadFactor = math.max((specMotor.smoothedLoadPercentage or 0) * rpmPct, 0)
    local motorFactor = 0.5 * (0.2 * rpmFactor + 1.8 * loadFactor)
    local usageFactor = 1.5
    local mi = g_currentMission and g_currentMission.missionInfo
    if mi and mi.fuelUsage == 1 then usageFactor = 1.0 elseif mi and mi.fuelUsage == 3 then usageFactor = 2.5 end
    local consumer = getConsumer(specMotor, {"DIESEL","METHANE"})
    if consumer == nil then return nil end
    local usage = math.abs(consumer.usage or 0)
    if usage == 0 then return nil end
    local used = usageFactor * motorFactor * usage * dtMs
    local lph = (used / dtMs) * 1000.0 * 3600.0
    return math.max(0.0, lph)
end

-- Corner anchor (bottom-right), accounting for safe-frame and UI scale
local function getCornerAnchor(size)
    local xOff = _G["g_safeFrameOffsetX"] or 0.0
    local yOff = _G["g_safeFrameOffsetY"] or 0.0
    local right = 1.0 - xOff
    local bottom = yOff

    local uiScale = getUiScale() or 1.0
    local px = (FuelConsumptionHUD.UI_CORNER_OFFSET_PX.x or 0) * uiScale
    local py = (FuelConsumptionHUD.UI_CORNER_OFFSET_PX.y or 0) * uiScale

    local dx, dy = 0.0, 0.0
    if getNormalizedScreenValues then
        local ok, nx, ny = pcall(function() return getNormalizedScreenValues(px, py) end)
        if ok and nx and ny then
            dx, dy = nx, ny
        end
    end

    -- We render with vertical alignment TOP; to keep bottom spacing constant, anchor y = bottom + offset + text size
    return right - dx, bottom + dy + (size or 0)
end

-- Labels
local function formatLabelLiquid(lph) return string.format("%.1f l/h", lph) end
local function formatLabelElectricKw(kW) return string.format("%.1f kW", kW) end

function FuelConsumptionHUD:update(dt)
    if g_client == nil or g_currentMission == nil then
        S.visible = false
        return
    end

    S.accRefresh = S.accRefresh + dt
    if S.accRefresh < FuelConsumptionHUD.REFRESH_MS then return end
    S.accRefresh = S.accRefresh - FuelConsumptionHUD.REFRESH_MS

    if not FuelConsumptionHUD.SHOW_WITH_HUD_HIDDEN then
        if g_currentMission.hud and g_currentMission.hud.getIsVisible and not g_currentMission.hud:getIsVisible() then
            S.visible = false
            return
        end
    end

    local veh = getCurrentVehicle()
    if veh == nil then
        S.visible = false
        S.lastVehicle = nil
        S.energyType = "OTHER"
        S.fuelUnitIndex = nil
        S.dieselText = nil
        S.lastDieselShownT = 0.0
        S.wasMotorOn = false
        S.lvlSampleLevel = nil
        S.lvlSampleTime = nil
        S.dieselRateLph = 0.0
        return
    end

    local root = veh.rootVehicle or veh
    if S.lastVehicle ~= root then
        S.lastVehicle = root
        S.fuelUnitIndex = nil
        S.dieselText = nil
        S.lastDieselShownT = 0.0
        S.wasMotorOn = false
        S.lvlSampleLevel = nil
        S.lvlSampleTime = nil
        S.dieselRateLph = 0.0
    end

    local level, energyType, idx, specMotor, specFill = getEnergyLevelAndTypeWithIndex(root)
    S.energyType = energyType
    if idx ~= nil then S.fuelUnitIndex = idx end

    -- Hide HUD for any non-combustion and non-electric vehicles
    if energyType ~= "LIQUID" and energyType ~= "ELECTRIC" then
        S.visible = false
        return
    end

    if energyType == "LIQUID" then
        local speedKmh = getSpeedKmh(veh)
        local motorOn = isMotorLikelyRunning(specMotor, veh)
        local transitioned = (motorOn ~= S.wasMotorOn)
        S.wasMotorOn = motorOn
        if transitioned then
            S.lastDieselShownT = 0.0
            if not motorOn then
                S.lvlSampleLevel = level
                S.lvlSampleTime = nowSec()
                S.dieselRateLph = 0.0
            end
        end

        local lphFromFill = updateFillDeltaEstimator(level)
        local src = "none"

        local lph = 0.0
        if motorOn then
            -- Prefer GIANTS' lastFuelUsage, with outlier cap
            local lf = tonumber(specMotor and specMotor.lastFuelUsage)
            if lf and lf == lf and lf > 0 and lf <= FuelConsumptionHUD.DIESEL_MAX_VALID_LPH then
                lph = lf
                src = "lf"
            end
            -- Fallback: consumers
            if (src == "none") then
                local fromCons = computeLiquidUsageFromConsumers(specMotor, dt)
                if fromCons ~= nil and fromCons == fromCons and fromCons > 0 then
                    lph = math.min(fromCons, FuelConsumptionHUD.DIESEL_MAX_VALID_LPH)
                    src = "cons"
                end
            end
            -- Last resort: fill-level estimator
            if (src == "none") and lphFromFill and lphFromFill > FuelConsumptionHUD.DIESEL_DEADBAND_LPH then
                lph = math.min(lphFromFill, FuelConsumptionHUD.DIESEL_MAX_VALID_LPH)
                src = "fill"
            end
            -- Idle floor
            if lph > 0 and lph < FuelConsumptionHUD.DIESEL_IDLE_FLOOR_LPH then
                lph = FuelConsumptionHUD.DIESEL_IDLE_FLOOR_LPH
            end
        else
            lph = 0.0
        end

        lph = clamp(math.abs(lph), 0.0, FuelConsumptionHUD.DIESEL_MAX_VALID_LPH)

        local tNow = nowSec()
        if S.dieselText == nil or (tNow - (S.lastDieselShownT or 0)) >= FuelConsumptionHUD.DIESEL_DISPLAY_PERIOD_SEC then
            S.dieselText = formatLabelLiquid(lph)
            S.lastDieselShownT = tNow
        end
        S.text = S.dieselText
        S.visible = true

    elseif energyType == "ELECTRIC" then
        local speedKmh = getSpeedKmh(veh)
        local kW = 0.0
        do
            -- reuse estimator for electric level deltas (kWh/h == kW)
            local backupLevel, backupTime, backupRate = S.lvlSampleLevel, S.lvlSampleTime, S.dieselRateLph
            kW = updateFillDeltaEstimator(level)
            S.lvlSampleLevel, S.lvlSampleTime, S.dieselRateLph = backupLevel, backupTime, backupRate
        end
        if (kW < FuelConsumptionHUD.ELECTRIC_DEADBAND_KW) then
            local kWFromConsumers = computeElectricUsageFromConsumers(root, specMotor, dt)
            if kWFromConsumers ~= nil and kWFromConsumers == kWFromConsumers then
                kW = kWFromConsumers
            end
        end
        if speedKmh < FuelConsumptionHUD.ELECTRIC_IDLE_SPEED_KMH then
            kW = 0.0
        end
        kW = math.max(0.0, kW)
        S.text = formatLabelElectricKw(kW)
        S.visible = true
    end
end

function FuelConsumptionHUD:draw()
    if g_client == nil or not S.visible then return end

    -- Compute text size (respect UI scale as configured)
    local size = FuelConsumptionHUD.UI_SIZE
    if getCorrectTextSize then
        size = getCorrectTextSize(FuelConsumptionHUD.UI_SIZE)
    end
    if not FuelConsumptionHUD.UI_ENGINE_APPLIES_UI_SCALE then
        size = size * (getUiScale() or 1.0)
    end

    -- Compute bottom-right anchor after knowing size (for correct bottom spacing with top-aligned text)
    local x, y = getCornerAnchor(size)

    if setTextAlignment and setTextVerticalAlignment and renderText then
        setTextAlignment(RenderText.ALIGN_RIGHT)
        setTextVerticalAlignment(RenderText.VERTICAL_ALIGN_TOP)

        -- NO SHADOW: draw only the main text
        setTextColor(1, 1, 1, 1)
        renderText(x, y, size, S.text)

        -- restore defaults
        setTextColor(1, 1, 1, 1)
        setTextAlignment(RenderText.ALIGN_LEFT)
        setTextVerticalAlignment(RenderText.VERTICAL_ALIGN_BASELINE)
    end
end

addModEventListener(FuelConsumptionHUD)