-- FuelConsumptionHUD.lua
-- Robust Fuel & Electric power HUD
-- - Dynamic EMA smoothing for ECU/electric
-- - Robust fill-delta estimator (median + trimmed-mean + stability check)
-- - Plausibility filtering, caps and display '+' when capped
-- Text size follows in-game UI scale (SCALE_TEXT_WITH_UI = true)
-- Quantizes text size and pixel-snaps draw coordinates to reduce per-vehicle rasterization differences.

FuelConsumptionHUD = {
    REFRESH_MS = 100.0,
    DIESEL_DISPLAY_PERIOD_SEC = 1.0,

    -- Electric
    ELECTRIC_IDLE_SPEED_KMH = 0.1, -- compatibility retained
    ELECTRIC_DEADBAND_KW = 0.1,

    -- Diesel
    DIESEL_DEADBAND_LPH = 0.05,
    DIESEL_IDLE_FLOOR_LPH = 0.1,
    DIESEL_MAX_VALID_LPH = 2500.0,

    -- Fill-level delta estimator anti-spike
    FILL_DELTA_MIN_ELAPSED_SEC = 1.0,
    FILL_DELTA_MIN_CHANGE_L = 0.05,
    FILL_DELTA_SMOOTH_ALPHA = 0.25,
    FILL_DELTA_MEDIAN_BUF = 7,
    FILL_TRIM_RATIO = 0.25,
    FILL_MAX_ACCEPTABLE_COV = 0.6,
    FILL_NOISY_ALPHA_FACTOR = 0.25,

    FILL_IMPLAUSIBLE_RATIO = 0.5,

    -- LastFuel smoothing
    LASTFUEL_SMOOTH_ALPHA = 0.25,
    LASTFUEL_SMOOTH_BASE_ALPHA = 0.18,
    LASTFUEL_SMOOTH_GAIN = 0.6,
    LASTFUEL_SMOOTH_MAX_ALPHA = 0.75,
    LASTFUEL_MAX_STEP_LPH = 250.0,

    -- Electric smoothing
    ELECTRIC_SMOOTH_BASE_ALPHA = 0.18,
    ELECTRIC_SMOOTH_GAIN = 0.6,
    ELECTRIC_SMOOTH_MAX_ALPHA = 0.75,

    -- UI
    UI_SIZE = 0.014,
    UI_CORNER_OFFSET_PX = { x = 10, y = 186 },

    -- When true the HUD text size will be multiplied by the in-game UI scale (if getCorrectTextSize not present)
    SCALE_TEXT_WITH_UI = true,
    UI_SCALE_ADJUST = 1.0, -- multiply final text size by this (0.9 = 90% of engine/ui size)
    UI_OFFSET_SCALE = 0.97, -- how strongly offsets follow uiScale; 1.0 = follow fully, 0 = keep baseline
    SHOW_WITH_HUD_HIDDEN = false,

    -- Dev testing only
    DEBUG = false,
}

local S = {
    accRefresh = 0.0,
    visible = false,
    text = "",

    lastVehicle = nil,

    dieselText = nil,
    lastDieselShownT = 0.0,

    energyType = "OTHER",
    fuelUnitIndex = nil,
    wasMotorOn = false,

    -- fill estimator
    lvlSampleLevel = nil,
    lvlSampleTime = nil,
    dieselRateLph = 0.0,

    fillInstSamples = nil,

    lastFuelUsageSmoothed = nil,
    lastFuelUsagePrev = nil,
    lastFuelUsageTime = nil,

    electricUsageSmoothed = nil,
}

local function nowSec() return (g_time or 0) * 0.001 end
local function clamp(x, a, b) return math.max(a, math.min(b, x)) end
local function clamp01(x) return clamp(x, 0, 1) end

-- statistics helpers
local function mean(arr)
    if not arr or #arr == 0 then return nil end
    local s = 0.0
    for i=1,#arr do s = s + arr[i] end
    return s / #arr
end

local function stddev(arr, mu)
    if not arr or #arr == 0 then return nil end
    mu = mu or mean(arr)
    local s = 0.0
    for i=1,#arr do local d = arr[i]-mu; s = s + d*d end
    return math.sqrt(s / #arr)
end

local function medianOfArray(arr)
    if not arr or #arr == 0 then return nil end
    local tmp = {}
    for i = 1, #arr do tmp[i] = arr[i] end
    table.sort(tmp)
    local n = #tmp
    if n % 2 == 1 then
        return tmp[(n+1)/2]
    else
        return 0.5 * (tmp[n/2] + tmp[n/2 + 1])
    end
end

local function trimmedMean(arr, trimRatio)
    if not arr or #arr == 0 then return nil end
    trimRatio = trimRatio or 0.25
    local tmp = {}
    for i = 1, #arr do tmp[i] = arr[i] end
    table.sort(tmp)
    local n = #tmp
    local trim = math.floor(n * trimRatio + 0.5)
    local startIdx = 1 + trim
    local endIdx = n - trim
    if startIdx > endIdx then
        return mean(tmp)
    end
    local s = 0.0
    local count = 0
    for i = startIdx, endIdx do s = s + tmp[i]; count = count + 1 end
    return (count > 0) and (s / count) or nil
end

local function pushFillInstSample(val)
    if val == nil then return end
    S.fillInstSamples = S.fillInstSamples or {}
    table.insert(S.fillInstSamples, val)
    local maxN = FuelConsumptionHUD.FILL_DELTA_MEDIAN_BUF or 7
    while #S.fillInstSamples > maxN do table.remove(S.fillInstSamples, 1) end
end

local function resetPerVehicleState()
    S.fuelUnitIndex = nil
    S.dieselText = nil
    S.lastDieselShownT = 0.0
    S.wasMotorOn = false
    S.lvlSampleLevel = nil
    S.lvlSampleTime = nil
    S.dieselRateLph = 0.0
    S.energyType = "OTHER"

    S.lastFuelUsageSmoothed = nil
    S.lastFuelUsagePrev = nil
    S.lastFuelUsageTime = nil
    S.electricUsageSmoothed = nil

    S.fillInstSamples = {}
end

local function getUiScale()
    local scale = 1.0
    if g_gameSettings ~= nil then
        if g_gameSettings.getValue ~= nil then
            local ok, val = pcall(function() return g_gameSettings:getValue("uiScale") end)
            if ok and type(val) == "number" and val > 0 then scale = val end
        end
        if scale == 1.0 and type(g_gameSettings.uiScale) == "number" and g_gameSettings.uiScale > 0 then scale = g_gameSettings.uiScale end
    end
    if scale == 1.0 and type(_G["g_uiScale"]) == "number" and _G["g_uiScale"] > 0 then scale = _G["g_uiScale"] end
    return scale
end

local function getCurrentVehicle()
    local hud = g_currentMission and g_currentMission.hud
    local speedo = hud and hud.speedMeter
    if speedo and speedo.vehicle ~= nil then return speedo.vehicle end
    if hud and hud.activeDisplayVehicle ~= nil then return hud.activeDisplayVehicle end
    if g_currentMission and g_currentMission.controlledVehicle ~= nil then return g_currentMission.controlledVehicle end
    if g_currentMission and g_currentMission.vehicles then
        for _, v in ipairs(g_currentMission.vehicles) do
            if v then
                if v.getIsControlled and v:getIsControlled() then return v end
                if v.getIsEntered and v:getIsEntered() then return v end
                local spec = v.spec_enterable
                if spec and (spec.isControlled or spec.entered or (spec.numPassengers or 0) > 0) then
                    return v
                end
            end
        end
    end
    return nil
end

local function resolveFillUnitIndex(root, ft)
    if root == nil or ft == nil then return nil end
    if root.getConsumerFillUnitIndex ~= nil then
        local ok, idx = pcall(function() return root:getConsumerFillUnitIndex(ft) end)
        if ok and idx ~= nil then return idx end
    end
    return nil
end

local function getEnergyLevelAndTypeWithIndex(veh)
    if veh == nil then return nil, "OTHER", nil, nil, nil end
    local root = veh.rootVehicle or veh
    local specFill = root.spec_fillUnit
    local specMotor = root.spec_motorized
    if specFill == nil or specFill.fillUnits == nil then
        return nil, "OTHER", nil, specMotor, specFill
    end

    local FT = _G["FillType"] or {}
    local idx

    if FT.ELECTRICCHARGE then
        idx = resolveFillUnitIndex(root, FT.ELECTRICCHARGE)
        if idx and specFill.fillUnits[idx] then
            return specFill.fillUnits[idx].fillLevel, "ELECTRIC", idx, specMotor, specFill
        end
    end
    if FT.ELECTRICITY then
        idx = resolveFillUnitIndex(root, FT.ELECTRICITY)
        if idx and specFill.fillUnits[idx] then
            return specFill.fillUnits[idx].fillLevel, "ELECTRIC", idx, specMotor, specFill
        end
    end
    if FT.DIESEL then
        idx = resolveFillUnitIndex(root, FT.DIESEL)
        if idx and specFill.fillUnits[idx] then
            return specFill.fillUnits[idx].fillLevel, "LIQUID", idx, specMotor, specFill
        end
    end
    if FT.METHANE then
        idx = resolveFillUnitIndex(root, FT.METHANE)
        if idx and specFill.fillUnits[idx] then
            return specFill.fillUnits[idx].fillLevel, "LIQUID", idx, specMotor, specFill
        end
    end

    local motorIdx = specMotor and specMotor.fuelFillUnitIndex or nil
    if motorIdx and specFill.fillUnits[motorIdx] then
        local fu = specFill.fillUnits[motorIdx]
        if (FT.ELECTRICCHARGE and fu.fillType == FT.ELECTRICCHARGE) or (FT.ELECTRICITY and fu.fillType == FT.ELECTRICITY) then
            return fu.fillLevel, "ELECTRIC", motorIdx, specMotor, specFill
        end
        if (FT.DIESEL and fu.fillType == FT.DIESEL) or (FT.METHANE and fu.fillType == FT.METHANE) then
            return fu.fillLevel, "LIQUID", motorIdx, specMotor, specFill
        end
    end

    for i, fu in ipairs(specFill.fillUnits) do
        if fu then
            if ((FT.ELECTRICCHARGE and fu.fillType == FT.ELECTRICCHARGE) or (FT.ELECTRICITY and fu.fillType == FT.ELECTRICITY)) then
                return fu.fillLevel, "ELECTRIC", i, specMotor, specFill
            end
        end
    end
    for i, fu in ipairs(specFill.fillUnits) do
        if fu then
            if ((FT.DIESEL and fu.fillType == FT.DIESEL) or (FT.METHANE and fu.fillType == FT.METHANE)) then
                return fu.fillLevel, "LIQUID", i, specMotor, specFill
            end
        end
    end

    return nil, "OTHER", nil, specMotor, specFill
end

local function getSpeedKmh(veh)
    local s = 0.0
    if veh and veh.getLastSpeed ~= nil then
        local ok, val = pcall(function() return veh:getLastSpeed() end)
        if ok and val then s = val end
    end
    return s or 0.0
end

local function isMotorStartedOnly(specMotor)
    if not specMotor then return false end
    if specMotor.isMotorStarted == true then return true end
    if specMotor.getIsMotorStarted ~= nil then
        local ok, started = pcall(function() return specMotor:getIsMotorStarted() end)
        if ok and started then return true end
    end
    local m = specMotor.motor
    if m then
        local lastR = (m.lastMotorRpm or 0)
        local minR = (m.minRpm or 0)
        if lastR > 0 and lastR >= minR then return true end
    end
    return false
end

local function isMotorLikelyRunning(specMotor, veh)
    if not specMotor then return false end
    if specMotor.isMotorStarted == true then return true end
    if specMotor.getIsMotorStarted ~= nil then
        local ok, started = pcall(function() return specMotor:getIsMotorStarted() end)
        if ok and started then return true end
    end
    local m = specMotor.motor
    if m then
        local lastR = (m.lastMotorRpm or 0)
        local minR = (m.minRpm or 0)
        if lastR > 0 and lastR >= minR then return true end
    end
    if getSpeedKmh(veh) > 0.2 then return true end
    return false
end

-- cached detection for getCorrectTextSize (performance)
local _fc_textSizeMode = nil -- "noHelper" | "engineScaled" | "engineNotScaled"
local _fc_getCorrectTextSize = getCorrectTextSize -- may be nil

local function fc_detectTextSizeMode(baseSize)
    if _fc_textSizeMode ~= nil then return _fc_textSizeMode end
    if not _fc_getCorrectTextSize then
        _fc_textSizeMode = "noHelper"
        return _fc_textSizeMode
    end

    local ok1, s1 = pcall(function() return _fc_getCorrectTextSize(baseSize) end)
    if not ok1 or not s1 or type(s1) ~= "number" or s1 <= 0 then
        _fc_textSizeMode = "noHelper"
        return _fc_textSizeMode
    end

    local uiScale = getUiScale() or 1.0
    local ok2, s2 = pcall(function() return _fc_getCorrectTextSize(baseSize * uiScale) end)
    if ok2 and s2 and type(s2) == "number" and s1 > 0 then
        local ratio = s2 / s1
        if math.abs(ratio - uiScale) < 0.02 then
            _fc_textSizeMode = "engineNotScaled"
        else
            _fc_textSizeMode = "engineScaled"
        end
    else
        _fc_textSizeMode = "engineScaled"
    end
    return _fc_textSizeMode
end

-- Fill-delta estimator (robust)
local function updateFillDeltaEstimator(level)
    local t = nowSec()
    if level == nil then
        S.lvlSampleLevel = nil
        S.lvlSampleTime = nil
        S.dieselRateLph = 0.0
        S.fillInstSamples = {}
        return 0.0
    end
    if S.lvlSampleLevel == nil or S.lvlSampleTime == nil then
        S.lvlSampleLevel = level
        S.lvlSampleTime = t
        S.dieselRateLph = 0.0
        S.fillInstSamples = {}
        return 0.0
    end

    local dLevel = S.lvlSampleLevel - level
    local elapsed = t - S.lvlSampleTime

    if math.abs(dLevel) >= FuelConsumptionHUD.FILL_DELTA_MIN_CHANGE_L and elapsed >= FuelConsumptionHUD.FILL_DELTA_MIN_ELAPSED_SEC then
        local used = math.max(0.0, dLevel)
        local instLph = (used / elapsed) * 3600.0
        instLph = math.max(0.0, instLph)
        if instLph > FuelConsumptionHUD.DIESEL_MAX_VALID_LPH then
            instLph = FuelConsumptionHUD.DIESEL_MAX_VALID_LPH
        end

        pushFillInstSample(instLph)

        local buf = S.fillInstSamples or {}
        local med = medianOfArray(buf) or instLph
        local tmean = trimmedMean(buf, FuelConsumptionHUD.FILL_TRIM_RATIO)
        local mu = tmean or med
        local sd = stddev(buf, mu) or 0.0
        local cov = (med > 0) and (sd / med) or 0.0

        local sampleVal = med
        local alpha = clamp01(FuelConsumptionHUD.FILL_DELTA_SMOOTH_ALPHA)

        if #buf >= 3 then
            if cov <= (FuelConsumptionHUD.FILL_MAX_ACCEPTABLE_COV or 0.6) then
                if tmean then sampleVal = tmean end
            else
                sampleVal = med
                alpha = alpha * (FuelConsumptionHUD.FILL_NOISY_ALPHA_FACTOR or 0.25)
            end
        else
            sampleVal = med
            alpha = alpha * 0.5
        end

        S.dieselRateLph = S.dieselRateLph * (1 - alpha) + sampleVal * alpha

        S.lvlSampleLevel = level
        S.lvlSampleTime = t

    elseif elapsed >= 5.0 then
        S.dieselRateLph = S.dieselRateLph * 0.8
        S.lvlSampleLevel = level
        S.lvlSampleTime = t
    end

    return S.dieselRateLph or 0.0
end

local function arrayContains(t, val)
    for _, v in ipairs(t) do if v == val then return true end end
    return false
end

local function getConsumer(specMotor, candidates)
    if not specMotor then return nil end
    local byName = specMotor.consumersByFillTypeName
    if byName ~= nil then
        for _, name in ipairs(candidates) do
            local c = byName[name]
            if c and (c.usage ~= nil) and c.usage ~= 0 then return c end
        end
    end
    local any = specMotor.consumers
    if type(any) == "table" then
        for _, c in pairs(any) do
            local n = c and (c.fillTypeName or c.name)
            if n and arrayContains(candidates, n) and (c.usage ~= nil) and c.usage ~= 0 then
                return c
            end
        end
        for _, c in pairs(any) do
            if c and (c.usage ~= nil) and c.usage ~= 0 then
                return c
            end
        end
    end
    return nil
end

local function computeElectricUsageFromConsumers(specMotor, dtMs)
    if specMotor == nil or dtMs == nil or dtMs <= 0 then return nil end
    local m = specMotor.motor
    if not m or m.minRpm == nil or m.maxRpm == nil then return nil end
    local rpmPct = (m.maxRpm ~= m.minRpm) and ((m.lastMotorRpm - m.minRpm) / (m.maxRpm - m.minRpm)) or 0
    rpmPct = clamp01(rpmPct)
    local idleFactor = 0.5
    local rpmFactor = idleFactor + rpmPct * (1 - idleFactor)
    local loadFactor = math.max((specMotor.smoothedLoadPercentage or 0) * rpmPct, 0)
    local motorFactor = 0.5 * (0.2 * rpmFactor + 1.8 * loadFactor)
    local usageFactor = 1.5
    local mi = g_currentMission and g_currentMission.missionInfo
    if mi and mi.fuelUsage == 1 then usageFactor = 1.0 elseif mi and mi.fuelUsage == 3 then usageFactor = 2.5 end
    local consumer = getConsumer(specMotor, {"ELECTRICCHARGE","ELECTRICITY"})
    if consumer == nil then return nil end
    local usage = math.abs(consumer.usage or 0)
    if usage == 0 then return nil end
    local used = usageFactor * motorFactor * usage * dtMs
    local kWhPerHour = (used / dtMs) * 1000.0 * 3600.0
    return kWhPerHour
end

local function computeLiquidUsageFromConsumers(specMotor, dtMs)
    if specMotor == nil or dtMs == nil or dtMs <= 0 then return nil end
    local m = specMotor.motor
    if not m or m.minRpm == nil or m.maxRpm == nil then return nil end
    local rpmPct = (m.maxRpm ~= m.minRpm) and ((m.lastMotorRpm - m.minRpm) / (m.maxRpm - m.minRpm)) or 0
    rpmPct = clamp01(rpmPct)
    local idleFactor = 0.5
    local rpmFactor = idleFactor + rpmPct * (1 - idleFactor)
    local loadFactor = math.max((specMotor.smoothedLoadPercentage or 0) * rpmPct, 0)
    local motorFactor = 0.5 * (0.2 * rpmFactor + 1.8 * loadFactor)
    local usageFactor = 1.5
    local mi = g_currentMission and g_currentMission.missionInfo
    if mi and mi.fuelUsage == 1 then usageFactor = 1.0 elseif mi and mi.fuelUsage == 3 then usageFactor = 2.5 end
    local consumer = getConsumer(specMotor, {"DIESEL","METHANE"})
    if consumer == nil then return nil end
    local usage = math.abs(consumer.usage or 0)
    if usage == 0 then return nil end
    local used = usageFactor * motorFactor * usage * dtMs
    local lph = (used / dtMs) * 1000.0 * 3600.0
    return math.max(0.0, lph)
end

-- getCornerAnchor: offsets are pixel values; UI_OFFSET_SCALE controls interpolation
local function getCornerAnchor(size)
    local xOff = _G["g_safeFrameOffsetX"] or 0.0
    local yOff = _G["g_safeFrameOffsetY"] or 0.0
    local right = 1.0 - xOff
    local bottom = yOff

    local uiScale = getUiScale() or 1.0
    local scaleFactor = (1 + (uiScale - 1) * (FuelConsumptionHUD.UI_OFFSET_SCALE or 1.0))
    local px = (FuelConsumptionHUD.UI_CORNER_OFFSET_PX.x or 0) * scaleFactor
    local py = (FuelConsumptionHUD.UI_CORNER_OFFSET_PX.y or 0) * scaleFactor

    local dx, dy = 0.0, 0.0
    if getNormalizedScreenValues then
        local ok, nx, ny = pcall(function() return getNormalizedScreenValues(px, py) end)
        if ok and nx and ny then dx, dy = nx, ny end
    end

    return right - dx, bottom + dy + (size or 0)
end

-- Pixel-snap helper used for draw coordinates
local function pixelSnapNormalized(nx, ny)
    local screenW = _G["g_screenWidth"] or _G["g_screenWidthLocal"] or 1920
    local screenH = _G["g_screenHeight"] or _G["g_screenHeightLocal"] or 1080
    local px = math.floor((nx or 0) * screenW + 0.5)
    local py = math.floor((ny or 0) * screenH + 0.5)
    return (px / screenW), (py / screenH)
end

local function formatLabelLiquid(lph, cap)
    if cap and lph >= cap then
        return string.format("%.0f+ l/h", cap)
    else
        return string.format("%.1f l/h", lph)
    end
end

local function formatLabelElectricKw(kW)
    return string.format("%.1f kW", kW)
end

function FuelConsumptionHUD:update(dt)
    if g_client == nil or g_currentMission == nil then
        S.visible = false
        return
    end

    -- QUICK VEHICLE PRESENCE & ENTER/LEAVE HANDLING (cache current vehicle)
    local quickVeh = getCurrentVehicle()
    if quickVeh == nil then
        S.visible = false
        S.lastVehicle = nil
        resetPerVehicleState()
        return
    end

    local rootQuick = quickVeh.rootVehicle or quickVeh
    if S.lastVehicle ~= rootQuick then
        S.lastVehicle = rootQuick
        resetPerVehicleState()
        S.accRefresh = FuelConsumptionHUD.REFRESH_MS

        local _, quickType = getEnergyLevelAndTypeWithIndex(rootQuick)
        if quickType == "ELECTRIC" then
            S.text = formatLabelElectricKw(0.0)
            S.visible = true
        elseif quickType == "LIQUID" then
            S.text = formatLabelLiquid(0.0, FuelConsumptionHUD.DIESEL_MAX_VALID_LPH)
            S.visible = true
        else
            S.visible = false
        end
        return
    end

    -- Throttle main calculations
    S.accRefresh = S.accRefresh + dt
    if S.accRefresh < FuelConsumptionHUD.REFRESH_MS then return end
    S.accRefresh = S.accRefresh - FuelConsumptionHUD.REFRESH_MS

    if not FuelConsumptionHUD.SHOW_WITH_HUD_HIDDEN then
        local hud = g_currentMission.hud
        if hud and hud.getIsVisible and not hud:getIsVisible() then
            S.visible = false
            return
        end
    end

    local veh = quickVeh
    local root = veh.rootVehicle or veh

    local level, energyType, idx, specMotor, specFill = getEnergyLevelAndTypeWithIndex(root)
    S.energyType = energyType
    if idx ~= nil then S.fuelUnitIndex = idx end

    if energyType ~= "LIQUID" and energyType ~= "ELECTRIC" then
        S.visible = false
        return
    end

    -- estimation logic unchanged...
    if energyType == "LIQUID" then
        -- diesel estimation (unchanged)
        local speedKmh = getSpeedKmh(veh)
        local motorOn = isMotorLikelyRunning(specMotor, veh)
        local transitioned = (motorOn ~= S.wasMotorOn)
        S.wasMotorOn = motorOn
        if transitioned then
            S.lastDieselShownT = 0.0
            if not motorOn then
                S.lvlSampleLevel = level
                S.lvlSampleTime = nowSec()
                S.dieselRateLph = 0.0
                S.fillInstSamples = {}
            end
        end

        -- LASTFUEL handling ...
        local lphFromLastFuel = nil
        local rawLfSample = nil
        if specMotor and specMotor.lastFuelUsage ~= nil then
            local lf = tonumber(specMotor.lastFuelUsage)
            if lf and lf == lf and lf > 0 and lf <= FuelConsumptionHUD.DIESEL_MAX_VALID_LPH then
                local tNow = nowSec()
                local prevT = S.lastFuelUsageTime or tNow
                local dtSec = math.max(1e-6, tNow - prevT) -- seconds
                local maxStep = FuelConsumptionHUD.LASTFUEL_MAX_STEP_LPH * dtSec
                if S.lastFuelUsagePrev ~= nil then
                    local delta = lf - S.lastFuelUsagePrev
                    if math.abs(delta) > maxStep then
                        lf = S.lastFuelUsagePrev + (delta > 0 and 1 or -1) * maxStep
                    end
                end

                local alpha = FuelConsumptionHUD.LASTFUEL_SMOOTH_ALPHA
                if S.lastFuelUsageSmoothed ~= nil then
                    local relDelta = math.abs(lf - S.lastFuelUsageSmoothed) / math.max(1.0, S.lastFuelUsageSmoothed)
                    local dynAlpha = FuelConsumptionHUD.LASTFUEL_SMOOTH_BASE_ALPHA + FuelConsumptionHUD.LASTFUEL_SMOOTH_GAIN * relDelta
                    dynAlpha = clamp(dynAlpha, 0.0, FuelConsumptionHUD.LASTFUEL_SMOOTH_MAX_ALPHA or 0.9)
                    alpha = dynAlpha
                else
                    alpha = FuelConsumptionHUD.LASTFUEL_SMOOTH_BASE_ALPHA
                end

                if S.lastFuelUsageSmoothed == nil then
                    S.lastFuelUsageSmoothed = lf
                else
                    S.lastFuelUsageSmoothed = S.lastFuelUsageSmoothed * (1 - alpha) + lf * alpha
                end
                S.lastFuelUsagePrev = lf
                S.lastFuelUsageTime = tNow
                lphFromLastFuel = S.lastFuelUsageSmoothed
                rawLfSample = lf
            end
        end

        local lphFromConsumers = nil
        do
            local cons = computeLiquidUsageFromConsumers(specMotor, dt)
            if cons ~= nil and cons == cons and cons > 0.0 then
                lphFromConsumers = math.min(cons, FuelConsumptionHUD.DIESEL_MAX_VALID_LPH)
            end
        end

        local lphFromFill = nil
        do
            local fromFill = updateFillDeltaEstimator(level)
            if fromFill and fromFill > 0 then
                lphFromFill = math.min(fromFill, FuelConsumptionHUD.DIESEL_MAX_VALID_LPH)
            end
        end

        if lphFromLastFuel ~= nil and lphFromFill ~= nil then
            local rel = math.abs(lphFromFill - lphFromLastFuel) / math.max(1.0, lphFromLastFuel)
            if rel > (FuelConsumptionHUD.FILL_IMPLAUSIBLE_RATIO or 0.5) then
                lphFromFill = nil
            end
        end

        local lph = 0.0
        if motorOn then
            if lphFromLastFuel ~= nil then
                if lphFromConsumers ~= nil then
                    lph = 0.90 * lphFromLastFuel + 0.10 * lphFromConsumers
                else
                    lph = lphFromLastFuel
                end
            elseif lphFromConsumers ~= nil then
                if lphFromFill ~= nil then
                    lph = 0.75 * lphFromConsumers + 0.25 * lphFromFill
                else
                    lph = lphFromConsumers
                end
            elseif lphFromFill ~= nil and lphFromFill > FuelConsumptionHUD.DIESEL_DEADBAND_LPH then
                lph = lphFromFill
            else
                lph = 0.0
            end

            if lph > 0 and lph < FuelConsumptionHUD.DIESEL_IDLE_FLOOR_LPH then
                lph = FuelConsumptionHUD.DIESEL_IDLE_FLOOR_LPH
            end
        else
            lph = 0.0
        end

        lph = clamp(math.abs(lph or 0.0), 0.0, FuelConsumptionHUD.DIESEL_MAX_VALID_LPH)

        local tNow = nowSec()
        if S.dieselText == nil or (tNow - (S.lastDieselShownT or 0)) >= FuelConsumptionHUD.DIESEL_DISPLAY_PERIOD_SEC then
            S.dieselText = formatLabelLiquid(lph, FuelConsumptionHUD.DIESEL_MAX_VALID_LPH)
            S.lastDieselShownT = tNow
        end
        S.text = S.dieselText
        S.visible = true

    elseif energyType == "ELECTRIC" then
        local motorOn = isMotorStartedOnly(specMotor)
        local transitioned = (motorOn ~= S.wasMotorOn)
        S.wasMotorOn = motorOn
        if transitioned and not motorOn then
            S.lvlSampleLevel = level
            S.lvlSampleTime = nowSec()
            S.dieselRateLph = 0.0
            S.fillInstSamples = {}
            S.electricUsageSmoothed = nil
        end

        local kW = 0.0
        if motorOn then
            local kWFromFill = updateFillDeltaEstimator(level) or 0.0
            local kWFromConsumers = computeElectricUsageFromConsumers(specMotor, dt) or 0.0

            if kWFromConsumers ~= nil and kWFromConsumers > 0.0 then
                if kWFromFill and kWFromFill > 0.0 then
                    kW = 0.7 * kWFromConsumers + 0.3 * kWFromFill
                else
                    kW = kWFromConsumers
                end
            else
                kW = kWFromFill or 0.0
            end

            if S.electricUsageSmoothed == nil then
                S.electricUsageSmoothed = kW
            else
                local relDelta = math.abs(kW - S.electricUsageSmoothed) / math.max(1.0, S.electricUsageSmoothed)
                local dynAlpha = FuelConsumptionHUD.ELECTRIC_SMOOTH_BASE_ALPHA + FuelConsumptionHUD.ELECTRIC_SMOOTH_GAIN * relDelta
                dynAlpha = clamp(dynAlpha, 0.0, FuelConsumptionHUD.ELECTRIC_SMOOTH_MAX_ALPHA or 0.9)
                S.electricUsageSmoothed = S.electricUsageSmoothed * (1 - dynAlpha) + kW * dynAlpha
            end
            kW = math.max(0.0, S.electricUsageSmoothed or 0.0)
        else
            kW = 0.0
        end

        S.text = formatLabelElectricKw(kW)
        S.visible = true
    end
end

function FuelConsumptionHUD:draw()
    if g_client == nil or not S.visible then return end
    if not renderText or not setTextAlignment or not setTextVerticalAlignment then return end

    -- cache globals to locals for this frame
    local renderTextLocal = renderText
    local setTextAlignmentLocal = setTextAlignment
    local setTextVerticalAlignmentLocal = setTextVerticalAlignment

    -- detect text-size behaviour once and then use fast paths
    fc_detectTextSizeMode(FuelConsumptionHUD.UI_SIZE)
    local mode = _fc_textSizeMode

    local function computeTextSize(baseSize, scaleWithUI)
        local uiScale = getUiScale() or 1.0
        if mode == "noHelper" or not _fc_getCorrectTextSize then
            return baseSize * (scaleWithUI and uiScale or 1.0)
        elseif mode == "engineNotScaled" then
            return _fc_getCorrectTextSize(baseSize * (scaleWithUI and uiScale or 1.0))
        else
            return _fc_getCorrectTextSize(baseSize)
        end
    end

    local size = computeTextSize(FuelConsumptionHUD.UI_SIZE, FuelConsumptionHUD.SCALE_TEXT_WITH_UI)

    -- allow manual tweak and quantize
    size = size * (FuelConsumptionHUD.UI_SCALE_ADJUST or 1.0)
    size = math.floor((size or 0) * 2000 + 0.5) / 2000

    local x, y = getCornerAnchor(size)

    -- pixel-snap coordinates to reduce subpixel rendering variance
    x, y = pixelSnapNormalized(x, y)

    setTextAlignmentLocal(RenderText.ALIGN_RIGHT)
    setTextVerticalAlignmentLocal(RenderText.VERTICAL_ALIGN_TOP)

    setTextColor(1, 1, 1, 1)
    renderTextLocal(x, y, size, S.text)

    setTextAlignment(RenderText.ALIGN_LEFT)
    setTextVerticalAlignment(RenderText.VERTICAL_ALIGN_BASELINE)
end

addModEventListener(FuelConsumptionHUD)