--[[
	ToggleableBaleWrapper.lua
	
	Autor: 		Ifko[nator]
	Datum: 		24.11.2024
	Version: 	1.0
	
	Changelog:	v1.0 @24.11.2024 - initial implementation in FS 25
]]

ToggleableBaleWrapper = {};
ToggleableBaleWrapper.modName = "";

ToggleableBaleWrapper.DROP_MODE_MANUAL = 1;
ToggleableBaleWrapper.DROP_MODE_AUTOMATIC = 2;
ToggleableBaleWrapper.DROP_MODE_COLLECT = 3;
ToggleableBaleWrapper.DROP_MODE_MAX_STATE = 3;
ToggleableBaleWrapper.FORCE_DROP_BALE = false;

for _, mod in pairs(g_modManager.mods) do
	if mod.title:upper() == "RUNDBALLENPRESSEN ERWEITERUNG" or mod.title:upper() == "ROUND BALER EXTENSION" then		
		if g_modIsLoaded[tostring(mod.modName)] then	
			ToggleableBaleWrapper.modName = mod.modName;

			break;
		end;
	end;
end;

function ToggleableBaleWrapper.initSpecialization()
	local schemaSavegame = Vehicle.xmlSchemaSavegame;

	schemaSavegame:register(XMLValueType.INT, "vehicles.vehicle(?).toggleableBaleWrapper#currentDropModeState", "Current drop mode.", ToggleableBaleWrapper.DROP_MODE_MANUAL);
	schemaSavegame:register(XMLValueType.BOOL, "vehicles.vehicle(?).toggleableBaleWrapper#baleWrapperIsActive", "Is bale wrapper active.", true);
end;

function ToggleableBaleWrapper.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(BaleWrapper, specializations);
end;

function ToggleableBaleWrapper.registerEventListeners(vehicleType)
	local functionNames = {
		"onLoad",
		"onUpdate",
		"saveToXMLFile",
		"onWriteStream",
		"onReadStream",
		"onRegisterActionEvents"
	};

	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, ToggleableBaleWrapper);
	end;
end;

function ToggleableBaleWrapper.registerFunctions(vehicleType)
	local functionNames = {
		"setDropMode",
		"setBaleWrapperState"
	};

	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerFunction(vehicleType, functionName, ToggleableBaleWrapper[functionName]);
	end;
end;

function ToggleableBaleWrapper:onLoad(savegame)
	local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
	local specBaler = self.spec_baler;

	specToggleableBaleWrapper.isGoeweilBaler = Utils.getNoNil(self.configFileName:find("varioMaster.xml") or self.configFileName:find("ltMaster.xml"), false);

	specToggleableBaleWrapper.baleWrapperIsActive = true;

	specToggleableBaleWrapper.currentDropModeState = ToggleableBaleWrapper.DROP_MODE_MANUAL;
	specToggleableBaleWrapper.maxDropState = ToggleableBaleWrapper.DROP_MODE_MAX_STATE;

	specToggleableBaleWrapper.l10NTexts = {};

	if specBaler ~= nil then
		if savegame ~= nil and not savegame.resetVehicles then
			specToggleableBaleWrapper.baleWrapperIsActive = savegame.xmlFile:getValue(savegame.key .. ".toggleableBaleWrapper#baleWrapperIsActive", specToggleableBaleWrapper.baleWrapperIsActive);
			specToggleableBaleWrapper.currentDropModeState = savegame.xmlFile:getValue(savegame.key .. ".toggleableBaleWrapper#currentDropModeState", specToggleableBaleWrapper.currentDropModeState);
		end;

		if not specToggleableBaleWrapper.isGoeweilBaler then
			self:setDropMode(specToggleableBaleWrapper.currentDropModeState, false);
		end;

		self:setBaleWrapperState(specToggleableBaleWrapper.baleWrapperIsActive);

		self.doStateChange = Utils.overwrittenFunction(self.doStateChange, ToggleableBaleWrapper.doStateChange);
		self.pickupWrapperBale = Utils.overwrittenFunction(self.pickupWrapperBale, ToggleableBaleWrapper.pickupWrapperBale);

		local l10NTexts = {
			"BALE_WRAPPER_MODE_ACTIVATED",
			"BALE_WRAPPER_MODE_DEACTIVATED",
			"DROP_MODE_MANUALLY",
			"DROP_MODE_AUTOMATIC",
			"DROP_MODE_COLLECT",
			"DROP_MODE_CURRENT",
		};

		for _, l10NText in pairs(l10NTexts) do
			specToggleableBaleWrapper.l10NTexts[l10NText] = g_i18n:getText(l10NText, ToggleableBaleWrapper.modName);
		end;
	end;
end;

function ToggleableBaleWrapper:setDropMode(currentDropModeState, noEventSend)
    local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;

	if currentDropModeState ~= specToggleableBaleWrapper.currentDropModeState then
		specToggleableBaleWrapper.currentDropModeState = currentDropModeState;

		 if not noEventSend then
			if g_server ~= nil then
				g_server:broadcastEvent(SetDropModeEvent.new(self, currentDropModeState), nil, nil, self);
			else
				g_client:getServerConnection():sendEvent(SetDropModeEvent.new(self, currentDropModeState));
			end;
		end;
	end;
end;

function ToggleableBaleWrapper:onWriteStream(streamId, connection)
	if not connection:getIsServer() then 
		local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
		local specBaler = self.spec_baler;

		if specBaler ~= nil then	
			streamWriteUIntN(streamId, specToggleableBaleWrapper.currentDropModeState, specToggleableBaleWrapper.maxDropState);
			streamWriteBool(streamId, specToggleableBaleWrapper.baleWrapperIsActive);
		end;
	end;
end;

function ToggleableBaleWrapper:onReadStream(streamId, connection)
	if connection:getIsServer() then
		local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
		local specBaler = self.spec_baler;

		if specBaler ~= nil then
			specToggleableBaleWrapper.currentDropModeState = streamReadUIntN(streamId, specToggleableBaleWrapper.maxDropState);
			specToggleableBaleWrapper.baleWrapperIsActive = streamReadBool(streamId);
		end;
	end;
end;

function ToggleableBaleWrapper:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
	if self.isClient then
        local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
		local specBaler = self.spec_baler;
		local specBaleWrapper = self.spec_baleWrapper;

		self:clearActionEventsTable(specToggleableBaleWrapper.actionEvents);

		if self:getIsActiveForInput(true, false) and specBaler ~= nil then
            local newFunctions = {
				"CHANGE_DROP_MODE_BUTTON",
				"TOGGLE_BALE_WRAPPER_BUTTON"
			};

			for _, newFunction in ipairs(newFunctions) do
				local allowRegisterInput = true;

				if specToggleableBaleWrapper.isGoeweilBaler then
					allowRegisterInput = newFunction ~= "CHANGE_DROP_MODE_BUTTON";
				else
					if self.spec_baleWrapper ~= nil then
						actionEventAutomaticDropActivation = self.spec_baleWrapper.actionEvents[InputAction.IMPLEMENT_EXTRA4];
						actionEventManualDrop = self.spec_baleWrapper.actionEvents[InputAction.IMPLEMENT_EXTRA3];

						if actionEventAutomaticDropActivation ~= nil then
							--## deactivate default button for switching between automatic and manual unload for balers with bale wrappers
							
							g_inputBinding:setActionEventActive(actionEventAutomaticDropActivation.actionEventId, false);
						end;

						if actionEventManualDrop ~= nil then
							--## deactivate default button for manual drop if the collect mode is active
							
							g_inputBinding:setActionEventActive(actionEventManualDrop.actionEventId, specBaleWrapper.baleWrapperState == BaleWrapper.STATE_WRAPPER_FINSIHED and specToggleableBaleWrapper.currentDropModeState == ToggleableBaleWrapper.DROP_MODE_MANUAL);
						end;
					end;
				end;

				if allowRegisterInput then
					local _, actionEventId = self:addActionEvent(specToggleableBaleWrapper.actionEvents, InputAction[newFunction], self, ToggleableBaleWrapper.actionEventChangeDropeModeAndBaleWrapperState, false, true, false, true, nil);

					g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_VERY_HIGH);
					g_inputBinding:setActionEventActive(actionEventId, newFunction == "CHANGE_DROP_MODE_BUTTON");
				end;

				ToggleableBaleWrapper.updateActionEvents(self);
			end;
		end;
	end;
end;

function ToggleableBaleWrapper.actionEventChangeDropeModeAndBaleWrapperState(self, actionName, inputValue, callbackState, isAnalog)
	local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;

	if actionName == "CHANGE_DROP_MODE_BUTTON" then
		if specToggleableBaleWrapper.currentDropModeState < specToggleableBaleWrapper.maxDropState then
			self:setDropMode(specToggleableBaleWrapper.currentDropModeState + 1);
		else
			self:setDropMode(ToggleableBaleWrapper.DROP_MODE_MANUAL);
		end;
	else
		self:setBaleWrapperState(not specToggleableBaleWrapper.baleWrapperIsActive);
	end;
end;

function ToggleableBaleWrapper:doStateChange(superFunc, id, nearestBaleServerId)
    local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
	local specBaleWrapper = self.spec_baleWrapper;

	if id == BaleWrapper.CHANGE_WRAPPING_START and not specToggleableBaleWrapper.baleWrapperIsActive then
        specBaleWrapper.baleWrapperState = BaleWrapper.STATE_WRAPPER_FINSIHED;

		return;
    end;

	if id == BaleWrapper.CHANGE_WRAPPING_START or specBaleWrapper.baleWrapperState ~= BaleWrapper.STATE_WRAPPER_FINSIHED and id == BaleWrapper.CHANGE_WRAPPER_START_DROP_BALE then
		local bale = NetworkUtil.getObject(specBaleWrapper.currentWrapper.currentBale);
		local baleType = specBaleWrapper.currentWrapper.allowedBaleTypes[specBaleWrapper.currentBaleTypeIndex];

		if not bale:getSupportsWrapping() or baleType.skipWrapping or bale.wrappingState == 1 then
			if self.isServer then
				specBaleWrapper.setWrappingStateFinished = true;
			end;

			return;
		end;
	end;

	if id == BaleWrapper.CHANGE_GRAB_BALE then
		local bale = NetworkUtil.getObject(nearestBaleServerId);

		specBaleWrapper.baleGrabber.currentBale = nearestBaleServerId;

		if bale ~= nil then
			local x, y, z = localToLocal(bale.nodeId, getParent(specBaleWrapper.baleGrabber.grabNode), 0, 0, 0);

			setTranslation(specBaleWrapper.baleGrabber.grabNode, x, y, z)
			bale:mountKinematic(self, specBaleWrapper.baleGrabber.grabNode, 0, 0, 0, 0, 0, 0);
			bale:setCanBeSold(false);

			specBaleWrapper.baleToMount = nil;

			self:playMoveToWrapper(bale);
		else
			specBaleWrapper.baleToMount = {
				serverId = nearestBaleServerId,
				linkNode = specBaleWrapper.baleGrabber.grabNode,
				trans = {0, 0, 0},
				rot = {0, 0, 0}
			};
		end;

		specBaleWrapper.baleWrapperState = BaleWrapper.STATE_MOVING_BALE_TO_WRAPPER;
	elseif id == BaleWrapper.CHANGE_DROP_BALE_AT_GRABBER then
		local attachNode = specBaleWrapper.currentWrapper.baleNode;
		local bale = NetworkUtil.getObject(specBaleWrapper.baleGrabber.currentBale);

		if bale ~= nil then
			bale:mountKinematic(self, attachNode, 0, 0, 0, 0, 0, 0);
			bale:setCanBeSold(false);

			specBaleWrapper.baleToMount = nil;
		else
			specBaleWrapper.baleToMount = {
				serverId = specBaleWrapper.baleGrabber.currentBale,
				linkNode = attachNode,
				trans = {0, 0, 0},
				rot = {0, 0, 0}
			};
		end;

		self:updateWrapNodes(true, false, 0);

		specBaleWrapper.currentWrapper.currentBale = specBaleWrapper.baleGrabber.currentBale;
		specBaleWrapper.baleGrabber.currentBale = nil;

		if specBaleWrapper.currentWrapper.animations.moveToWrapper.animName ~= nil and specBaleWrapper.currentWrapper.animations.moveToWrapper.reverseAfterMove then
			self:playAnimation(specBaleWrapper.currentWrapper.animations.moveToWrapper.animName, -specBaleWrapper.currentWrapper.animations.moveToWrapper.animSpeed, nil, true);
		end;

		specBaleWrapper.baleWrapperState = BaleWrapper.STATE_MOVING_GRABBER_TO_WORK;
	elseif id == BaleWrapper.CHANGE_WRAPPING_START then
		specBaleWrapper.baleWrapperState = BaleWrapper.STATE_WRAPPER_WRAPPING_BALE;

		if self.isClient then
			g_soundManager:playSamples(specBaleWrapper.currentWrapper.samples.start);
			g_soundManager:playSamples(specBaleWrapper.currentWrapper.samples.wrap, 0, specBaleWrapper.currentWrapper.samples.start[1]);
		end;

		if specBaleWrapper.currentWrapper.animations.wrapBale.animName ~= nil then
			self:playAnimation(specBaleWrapper.currentWrapper.animations.wrapBale.animName, specBaleWrapper.currentWrapper.animations.wrapBale.animSpeed, nil, true);
		end;
	elseif id == BaleWrapper.CHANGE_WRAPPING_BALE_FINSIHED then
		if self.isClient then
			g_soundManager:stopSamples(specBaleWrapper.currentWrapper.samples.wrap);
			g_soundManager:stopSamples(specBaleWrapper.currentWrapper.samples.stop);

			if specBaleWrapper.currentWrapper.wrappingSoundEndTime == 1 then
				g_soundManager:playSamples(specBaleWrapper.currentWrapper.samples.stop);
			end;

			g_soundManager:stopSamples(specBaleWrapper.currentWrapper.samples.start);
		end;

		self:updateWrappingState(1, true);

		specBaleWrapper.baleWrapperState = BaleWrapper.STATE_WRAPPER_FINSIHED;

		local bale = NetworkUtil.getObject(specBaleWrapper.currentWrapper.currentBale);
		local baleType = specBaleWrapper.currentWrapper.allowedBaleTypes[specBaleWrapper.currentBaleTypeIndex];
		local skippedWrapping = not bale:getSupportsWrapping() or baleType.skipWrapping;

		if skippedWrapping or bale.wrappingState == 1 then
			self:updateWrappingState(0, true);
		end;

		if not skippedWrapping then
			local animation = specBaleWrapper.currentWrapper.animations.resetWrapping;

			if animation.animName ~= nil then
				self:playAnimation(animation.animName, animation.animSpeed, nil, true);
			end;
		end;
	elseif id == BaleWrapper.CHANGE_WRAPPER_START_DROP_BALE then
		self:updateWrapNodes(false, false, 0);

		if specBaleWrapper.currentWrapper.animations.dropFromWrapper.animName ~= nil then
			self:playAnimation(specBaleWrapper.currentWrapper.animations.dropFromWrapper.animName, specBaleWrapper.currentWrapper.animations.dropFromWrapper.animSpeed, nil, true);
		end;

		specBaleWrapper.baleWrapperState = BaleWrapper.STATE_WRAPPER_DROPPING_BALE;
	elseif id == BaleWrapper.CHANGE_WRAPPER_BALE_DROPPED then
		local bale = NetworkUtil.getObject(specBaleWrapper.currentWrapper.currentBale);

		if bale ~= nil then
			bale:unmountKinematic();
			bale:setCanBeSold(true);

			local baleType = specBaleWrapper.currentWrapper.allowedBaleTypes[specBaleWrapper.currentBaleTypeIndex];

			if bale:getSupportsWrapping() and not baleType.skipWrapping and specToggleableBaleWrapper.baleWrapperIsActive then
				local stats = g_currentMission:farmStats(self:getOwnerFarmId());
				local total = stats:updateStats("wrappedBales", 1);

				g_achievementManager:tryUnlock("WrappedBales", total);

				if bale.wrappingState < 1 then
					bale:setWrappingState(1);
				end;
			end;
		end;

		specBaleWrapper.lastDroppedBale = bale;
		specBaleWrapper.currentWrapper.currentBale = nil;
		specBaleWrapper.currentWrapper.currentTime = 0;

		if specBaleWrapper.currentWrapper.animations.resetAfterDrop.animName ~= nil then
			local dropAnimationName = specBaleWrapper.currentWrapper.animations.dropFromWrapper.animName;

			if self:getIsAnimationPlaying(dropAnimationName) then
				self:stopAnimation(dropAnimationName, true);
				self:setAnimationTime(dropAnimationName, 1, true, false);
			end;

			self:playAnimation(specBaleWrapper.currentWrapper.animations.resetAfterDrop.animName, specBaleWrapper.currentWrapper.animations.resetAfterDrop.animSpeed, nil, true);
		end;

		self:setBaleWrapperType(specBaleWrapper.currentWrapper == specBaleWrapper.roundBaleWrapper, specBaleWrapper.currentBaleTypeIndex);

		specBaleWrapper.baleWrapperState = BaleWrapper.STATE_WRAPPER_RESETTING_PLATFORM;
	elseif id == BaleWrapper.CHANGE_WRAPPER_PLATFORM_RESET then
		self:updateWrappingState(0);
		self:updateWrapNodes(false, true, 0);

		specBaleWrapper.baleWrapperState = BaleWrapper.STATE_NONE;
	elseif id == BaleWrapper.CHANGE_BUTTON_EMPTY then
		assert(self.isServer);

		if specBaleWrapper.baleWrapperState == BaleWrapper.STATE_WRAPPER_FINSIHED then
			g_server:broadcastEvent(BaleWrapperStateEvent.new(self, BaleWrapper.CHANGE_WRAPPER_START_DROP_BALE), true, nil, self);
		end;
	end;
end;

function ToggleableBaleWrapper:pickupWrapperBale(superFunc, bale, baleType)
    local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;

	if not specToggleableBaleWrapper.baleWrapperIsActive then 
        g_server:broadcastEvent(BaleWrapperStateEvent.new(self, BaleWrapper.CHANGE_GRAB_BALE, NetworkUtil.getObjectId(bale)), true, nil, self);
    else
        superFunc(self, bale, baleType);
    end;
end;

function ToggleableBaleWrapper:setBaleWrapperState(baleWrapperIsActive, noEventSend)
	local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;

	if baleWrapperIsActive ~= specToggleableBaleWrapper.baleWrapperIsActive then
		if not noEventSend then
			if g_server ~= nil then
				g_server:broadcastEvent(SetBaleWrapperStateEvent.new(self, baleWrapperIsActive), nil, nil, self);
			else
				g_client:getServerConnection():sendEvent(SetBaleWrapperStateEvent.new(self, baleWrapperIsActive));
			end;
		end;

		specToggleableBaleWrapper.baleWrapperIsActive = baleWrapperIsActive;
	end;
end;

function ToggleableBaleWrapper:saveToXMLFile(xmlFile, key)
	local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
	local specBaler = self.spec_baler;

	if specBaler ~= nil then	
		xmlFile:setValue(key .. "#currentDropModeState", specToggleableBaleWrapper.currentDropModeState);
		xmlFile:setValue(key .. "#baleWrapperIsActive", specToggleableBaleWrapper.baleWrapperIsActive);
	end;
end;

function ToggleableBaleWrapper:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	if self:getIsActiveForInput(true, false) then
		local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
		local specBaleWrapper = self.spec_baleWrapper;
		local specBaler = self.spec_baler;

		if specBaler ~= nil then
			if specBaleWrapper.baleToLoad == nil 
				and specBaleWrapper.baleWrapperState < BaleWrapper.STATE_MOVING_BALE_TO_WRAPPER or specBaleWrapper.baleWrapperState >= BaleWrapper.STATE_WRAPPER_WRAPPING_BALE
				and specToggleableBaleWrapper.baleWrapperIsActive and self.isServer
			then
				local bale = NetworkUtil.getObject(specBaleWrapper.currentWrapper.currentBale);
						
				if bale ~= nil and bale.wrappingState == 0 and specBaleWrapper.baleWrapperState == BaleWrapper.STATE_WRAPPER_FINSIHED then
					self:playMoveToWrapper(bale);
				end;
			end;

			ToggleableBaleWrapper.updateActionEvents(self);

			if specBaleWrapper.baleWrapperState == BaleWrapper.STATE_WRAPPER_FINSIHED then
				local allowDropBale = specToggleableBaleWrapper.currentDropModeState == ToggleableBaleWrapper.DROP_MODE_AUTOMATIC;
				local attacherVehicle = self:getAttacherVehicle();
				local allowForceDropBale = true;

				if attacherVehicle ~= nil then
					specRoundBalerExtension = attacherVehicle.spec_roundBalerExtension;

					if specRoundBalerExtension ~= nil then
						allowForceDropBale = specRoundBalerExtension.driveOffTimer >= specRoundBalerExtension.driveOffTimerMax;
					end;
				end;

				if specToggleableBaleWrapper.currentDropModeState ~= ToggleableBaleWrapper.DROP_MODE_COLLECT and ToggleableBaleWrapper.FORCE_DROP_BALE then
					ToggleableBaleWrapper.FORCE_DROP_BALE = false;
				end;
	
				if ToggleableBaleWrapper.FORCE_DROP_BALE and allowForceDropBale then
					allowDropBale = true;
	
					ToggleableBaleWrapper.FORCE_DROP_BALE = false;
				elseif specToggleableBaleWrapper.currentDropModeState == ToggleableBaleWrapper.DROP_MODE_COLLECT then
					if self.spec_fillUnit:getFillUnitFillLevel(1) == self.spec_fillUnit:getFillUnitCapacity(1) then
						allowDropBale = true;
	
						ToggleableBaleWrapper.FORCE_DROP_BALE = true;
					end;
				end;

				if self.isClient and allowDropBale then
					g_client:getServerConnection():sendEvent(BaleWrapperStateEvent.new(self, BaleWrapper.CHANGE_BUTTON_EMPTY));
				end;
			end;
		end;
	end;
end;

function ToggleableBaleWrapper:updateActionEvents(superFunc)
	local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
	local specBaleWrapper = self.spec_baleWrapper;
	local specBaler = self.spec_baler;
	
	local dropBaleManualButton = specBaleWrapper.actionEvents[InputAction.IMPLEMENT_EXTRA3];
	local toggleBaleWrapperButton = specToggleableBaleWrapper.actionEvents[InputAction.TOGGLE_BALE_WRAPPER_BUTTON];
	local changeDropModeButton = specToggleableBaleWrapper.actionEvents[InputAction.CHANGE_DROP_MODE_BUTTON];

	if toggleBaleWrapperButton ~= nil then
		local currentText = specToggleableBaleWrapper.l10NTexts.BALE_WRAPPER_MODE_DEACTIVATED;

		if specToggleableBaleWrapper.baleWrapperIsActive then
			currentText = specToggleableBaleWrapper.l10NTexts.BALE_WRAPPER_MODE_ACTIVATED;
		end;

		local allowSetBaleWrapperState = specBaleWrapper.currentWrapper.currentBale == nil;
		local currentFillType = g_fillTypeManager:getFillTypeByIndex(self:getFillUnitFillType(specBaler.fillUnitIndex)).name;

		if currentFillType ~= nil and currentFillType ~= "UNKNOWN" then
			allowSetBaleWrapperState = currentFillType == "GRASS_WINDROW";
		end;

		g_inputBinding:setActionEventActive(toggleBaleWrapperButton.actionEventId, allowSetBaleWrapperState);
		g_inputBinding:setActionEventTextVisibility(toggleBaleWrapperButton.actionEventId, allowSetBaleWrapperState);
		g_inputBinding:setActionEventText(toggleBaleWrapperButton.actionEventId, currentText);
	end;

	if changeDropModeButton ~= nil then
		local currentText = specToggleableBaleWrapper.l10NTexts.DROP_MODE_MANUALLY;

		g_inputBinding:setActionEventActive(changeDropModeButton.actionEventId, not specToggleableBaleWrapper.isGoeweilBaler);

		if specToggleableBaleWrapper.currentDropModeState == ToggleableBaleWrapper.DROP_MODE_AUTOMATIC then
			currentText = specToggleableBaleWrapper.l10NTexts.DROP_MODE_AUTOMATIC;
		elseif specToggleableBaleWrapper.currentDropModeState == ToggleableBaleWrapper.DROP_MODE_COLLECT then
			currentText = specToggleableBaleWrapper.l10NTexts.DROP_MODE_COLLECT;
		end;

		g_inputBinding:setActionEventText(changeDropModeButton.actionEventId, specToggleableBaleWrapper.l10NTexts.DROP_MODE_CURRENT:gsub("currentDropMode", currentText));
	end;

	if dropBaleManualButton ~= nil then
		g_inputBinding:setActionEventActive(dropBaleManualButton.actionEventId, specBaleWrapper.baleWrapperState == BaleWrapper.STATE_WRAPPER_FINSIHED and specToggleableBaleWrapper.currentDropModeState == ToggleableBaleWrapper.DROP_MODE_MANUAL);
		g_inputBinding:setActionEventText(dropBaleManualButton.actionEventId, specBaleWrapper.currentWrapper.unloadBaleText);
	end;

	if specBaleWrapper.toggleableAutomaticDrop then
		actionEvent = specBaleWrapper.actionEvents[InputAction.IMPLEMENT_EXTRA4];

		if actionEvent ~= nil then
			g_inputBinding:setActionEventText(actionEvent.actionEventId, specBaleWrapper.automaticDrop and specBaleWrapper.toggleAutomaticDropTextNeg or specBaleWrapper.toggleAutomaticDropTextPos);
		end;
	end;
end;

BaleWrapper.updateActionEvents = Utils.overwrittenFunction(BaleWrapper.updateActionEvents, ToggleableBaleWrapper.updateActionEvents)

-----------------------------------------------------------------------------
--## Multiplayer Event | Change Drop Mode
-----------------------------------------------------------------------------

SetDropModeEvent = {};
SetDropModeEvent_mt = Class(SetDropModeEvent, Event);

InitEventClass(SetDropModeEvent, "SetDropModeEvent");

function SetDropModeEvent.emptyNew()
    local self = Event.new(SetDropModeEvent_mt);

    self.className = "SetDropModeEvent";

    return self;
end;

function SetDropModeEvent.new(baleWrapper, currentDropModeState)
    local self = SetDropModeEvent.emptyNew();
	
    self.baleWrapper = baleWrapper;
    self.currentDropModeState = currentDropModeState;
    
	return self;
end;

function SetDropModeEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.baleWrapper);

    streamWriteInt32(streamId, self.currentDropModeState);
end;

function SetDropModeEvent:readStream(streamId, connection)
    self.baleWrapper = NetworkUtil.readNodeObject(streamId);
    self.currentDropModeState = streamReadInt32(streamId);

    self:run(connection);
end;

function SetDropModeEvent:run(connection)	
	if not connection:getIsServer() then
		g_server:broadcastEvent(SetDropModeEvent.new(self.baleWrapper, self.currentDropModeState), nil, connection, self.baleWrapper);
	end;
	
	if self.baleWrapper ~= nil and self.baleWrapper.setDropMode ~= nil then	
		self.baleWrapper:setDropMode(self.currentDropModeState, true);
	end;
end;

-----------------------------------------------------------------------------
--## Multiplayer Event | Turn off/on bale wrapper
-----------------------------------------------------------------------------

SetBaleWrapperStateEvent = {};
SetBaleWrapperStateEvent_mt = Class(SetBaleWrapperStateEvent, Event);

InitEventClass(SetBaleWrapperStateEvent, "SetBaleWrapperStateEvent");

function SetBaleWrapperStateEvent.emptyNew()
	local self = Event.new(SetBaleWrapperStateEvent_mt);
    
	return self;
end;

function SetBaleWrapperStateEvent.new(baleWrapper, baleWrapperIsActive)
	local self = SetBaleWrapperStateEvent.emptyNew();
	
	self.baleWrapper = baleWrapper;
	self.baleWrapperIsActive = baleWrapperIsActive;
	
	return self;
end;

function SetBaleWrapperStateEvent:readStream(streamId, connection)
	self.baleWrapper = NetworkUtil.readNodeObject(streamId);
	self.baleWrapperIsActive = streamReadBool(streamId);
    
	self:run(connection);
end;

function SetBaleWrapperStateEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.baleWrapper);

	streamWriteBool(streamId, self.baleWrapperIsActive);
end;

function SetBaleWrapperStateEvent:run(connection)
	if not connection:getIsServer() then
		g_server:broadcastEvent(SetBaleWrapperStateEvent.new(self.baleWrapper, self.baleWrapperIsActive), nil, connection, self.baleWrapper);
	end;
	
    if self.baleWrapper ~= nil and self.baleWrapper.setBaleWrapperState ~= nil then
        self.baleWrapper:setBaleWrapperState(self.baleWrapperIsActive, true);
	end;
end;