---@class AutoFOVSettingsRepository
---This class is responsible for reading and writing settings
AutoFOVSettingsRepository = {
	MAIN_KEY = "AutoFOV",
	TARGET_FOV_KEY = "targetFOV",
    TARGET_FOV_INTERIOR_KEY = "targetFOVInterior",
    HARVESTER_FOV_INTERIOR_KEY = "harvesterFOVInterior"
}

---Writes the settings to our own XML file
---@param settings AutoFOVSettings @The settings object
function AutoFOVSettingsRepository.storeSettings(settings)
	local xmlPath = AutoFOVSettingsRepository.getXmlFilePath()
	if xmlPath == nil then
		return
	end

    -- Ensure directory exists
    local dir = getUserProfileAppPath() .. "modSettings"
    createFolder(dir)

	-- Create an empty XML file in memory
	local xmlFileId = createXMLFile("AutoFOV", xmlPath, AutoFOVSettingsRepository.MAIN_KEY)

	-- Add XML data in memory
	setXMLInt(xmlFileId, AutoFOVSettingsRepository.getPathForValue(AutoFOVSettingsRepository.TARGET_FOV_KEY), settings.targetFOV)
    setXMLInt(xmlFileId, AutoFOVSettingsRepository.getPathForValue(AutoFOVSettingsRepository.TARGET_FOV_INTERIOR_KEY), settings.targetFOVInterior)
    setXMLInt(xmlFileId, AutoFOVSettingsRepository.getPathForValue(AutoFOVSettingsRepository.HARVESTER_FOV_INTERIOR_KEY), settings.harvesterFOVInterior)

	-- Write the XML file to disk
	saveXMLFile(xmlFileId)
end

---Reads settings from an existing XML file, if such a file exists
---@param settings AutoFOVSettings @The settings object
function AutoFOVSettingsRepository.restoreSettings(settings)
	local xmlPath = AutoFOVSettingsRepository.getXmlFilePath()
	if xmlPath == nil then
		return
	end

	-- Abort if no settings have been saved yet
	if not fileExists(xmlPath) then
		return
	end

	-- Load the XML if possible
	local xmlFileId = loadXMLFile("AutoFOV", xmlPath)
	if xmlFileId == 0 then
		return
	end

	-- Read the values from memory
	settings.targetFOV = getXMLInt(xmlFileId, AutoFOVSettingsRepository.getPathForValue(AutoFOVSettingsRepository.TARGET_FOV_KEY)) or settings.targetFOV
    settings.targetFOVInterior = getXMLInt(xmlFileId, AutoFOVSettingsRepository.getPathForValue(AutoFOVSettingsRepository.TARGET_FOV_INTERIOR_KEY)) or settings.targetFOVInterior
    settings.harvesterFOVInterior = getXMLInt(xmlFileId, AutoFOVSettingsRepository.getPathForValue(AutoFOVSettingsRepository.HARVESTER_FOV_INTERIOR_KEY)) or settings.harvesterFOVInterior
	
	return
end

function AutoFOVSettingsRepository.getPathForValue(property, parentProperty)
	return ("%s.%s"):format(parentProperty or AutoFOVSettingsRepository.MAIN_KEY, property)
end

---Builds a path to the XML file.
---@return string|nil @The path to the XML file
function AutoFOVSettingsRepository.getXmlFilePath()
    return getUserProfileAppPath() .. "modSettings/FS25_AutoFOV.xml"
end
