-- AutoFOV.lua - FINAL PRODUCTION VERSION
-- Dynamic Refresh + Robust Binding Lookup + No Freeze

AutoFOV = {}
local MOD_NAME = g_currentModName

-- Initialize settings at file scope to ensure they exist for hooks
AutoFOV.settings = AutoFOVSettings.new()
AutoFOV.ui = AutoFOVUI.new(AutoFOV.settings)
AutoFOV.isSettingFov = false

function AutoFOV:loadMap(name)
    addConsoleCommand("gsSetFov", "Sets FOV for current vehicle (AutoFOV)", "consoleCommandFov", self)
end

function AutoFOV:deleteMap()
    removeConsoleCommand("gsSetFov")
end

function AutoFOV:mouseEvent(posX, posY, isDown, isUp, button)
end

function AutoFOV:draw()
end

function AutoFOV:update(dt)
end

function AutoFOV:onSetFov(actionName, inputValue, callbackState, isAnalog)
    self:trySetFov()
end

function AutoFOV.registerActionEvents(vehicle, superFunc, isActiveForInput, isActiveForInputIgnoreSelection)
    if superFunc ~= nil then
        superFunc(vehicle, isActiveForInput, isActiveForInputIgnoreSelection)
    end

    if isActiveForInputIgnoreSelection then
        -- Register the action event
        local _ , actionEventId = g_inputBinding:registerActionEvent(InputAction.SET_CUSTOM_FOV, AutoFOV, AutoFOV.onSetFov, false, true, false, true)
        if actionEventId then
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL)
            -- Hide from F1 context menu - FOV is set automatically, keybind is just for emergency use
            g_inputBinding:setActionEventTextVisibility(actionEventId, false)
        end
    end
end

function AutoFOV.onSetActiveCameraIndex(vehicle, superFunc, index)
    -- Recursion guard
    if AutoFOV.isSettingFov then
        return superFunc(vehicle, index)
    end

    local result = superFunc(vehicle, index)
    
    -- Automatically apply FOV when camera changes
    if AutoFOV then
        AutoFOV:trySetFov(vehicle)
    end
    
    return result
end

function AutoFOV:consoleCommandFov()
    self:trySetFov()
end

function AutoFOV:trySetFov(targetVehicle)
    if self.isSettingFov then return end
    self.isSettingFov = true

    -- 1. Get Vehicle
    local vehicle = targetVehicle
    if not vehicle then
        if g_currentMission and g_currentMission.controlledVehicle then
            vehicle = g_currentMission.controlledVehicle
        elseif g_localPlayer and g_localPlayer.getCurrentVehicle then
            vehicle = g_localPlayer:getCurrentVehicle()
        end
    end
    
    if not vehicle then 
        self.isSettingFov = false
        return 
    end
    
    -- 2. Get Camera Spec
    if not vehicle.spec_enterable or not vehicle.spec_enterable.cameras then 
        self.isSettingFov = false
        return 
    end
    local spec = vehicle.spec_enterable

    -- 3. Detect if vehicle is a harvester (has combine specialization)
    local isHarvester = vehicle.spec_combine ~= nil

    -- 4. Get FOV values from settings
    local targetFOVExterior = 30
    local targetFOVInterior = 60
    local harvesterFOVInterior = 75
    
    if self.settings then
        if self.settings.targetFOV then targetFOVExterior = self.settings.targetFOV end
        if self.settings.targetFOVInterior then targetFOVInterior = self.settings.targetFOVInterior end
        if self.settings.harvesterFOVInterior then harvesterFOVInterior = self.settings.harvesterFOVInterior end
    end

    local radFOVExterior = math.rad(targetFOVExterior)
    local radFOVInterior = math.rad(targetFOVInterior)
    local radHarvesterInterior = math.rad(harvesterFOVInterior)
    
    for i, cam in pairs(spec.cameras) do
        local targetRad = radFOVExterior
        if cam.isInside then
            -- Use harvester interior FOV if this is a combine, otherwise standard interior
            if isHarvester then
                targetRad = radHarvesterInterior
            else
                targetRad = radFOVInterior
            end
        end

        -- Set FOV on the camera data structure
        cam.fovY = targetRad
        cam.fovMin = targetRad
        cam.fovMax = targetRad
        
        -- Apply FOV directly to the camera node (native GIANTS Engine function)
        -- This updates the projection immediately without console spam
        if cam.cameraNode ~= nil and cam.cameraNode ~= 0 then
            setFovY(cam.cameraNode, targetRad)
        end
    end

    self.isSettingFov = false
end

-- Hooks for settings
BaseMission.loadMapFinished = Utils.prependedFunction(BaseMission.loadMapFinished, function(...)
    if AutoFOV.settings then
        AutoFOVSettingsRepository.restoreSettings(AutoFOV.settings)
        if AutoFOV.ui then
            AutoFOV.ui:injectUiSettings()
        end
    end
end)

-- Hook Enterable to register actions and camera switch
if Enterable then
    Enterable.onRegisterActionEvents = Utils.overwrittenFunction(Enterable.onRegisterActionEvents, AutoFOV.registerActionEvents)
    Enterable.setActiveCameraIndex = Utils.overwrittenFunction(Enterable.setActiveCameraIndex, AutoFOV.onSetActiveCameraIndex)
end

addModEventListener(AutoFOV)